/**************************************************************************************
 
   Copyright (c) Hilscher GmbH. All Rights Reserved.
 
 **************************************************************************************
 
   Filename:
    $Workfile: ToolkitSample.cpp $
   Last Modification:
    $Author: MichaelT $
    $Modtime: 6.11.09 15:57 $
    $Revision: 6224 $
   
   Targets:
     Win32/ANSI   : yes
     Win32/Unicode: yes (define _UNICODE)
     WinCE        : no
 
   Description:
       
   Changes:
 
     Version   Date        Author   Description
     ----------------------------------------------------------------------------------
     5         13.04.2011  SS       - Driver not opened in BlockDemo()
                                    - Wrong driver handle passed in ChannelDemo()
     4         05.11.2010  RM       Fixed ChannelDemo() / xChannelWrite() uses abRecvData[]
                                    instead of abSendData[]
     3         06.11.2009  RM		    - Slot number checking included
                                    - Read configuration file path creation fixed
     2         25.01.2008  MT		    ChannelDemo was callind IORead twice instead of Read/Write
     1         01.09.2006  RM       cifXGetMemptr() now requests the complete DPM of
                                    board 0 instead the memory of communication
                                    channel 0 ( -> CIFX_NO_CHANNEL)
     
**************************************************************************************/

#include "stdafx.h"
#include "OS_Dependent.h"
#include "cifXToolkit.h"
#include "CIFXErrors.h"

typedef int32_t(APIENTRY *PFN_DRIVER_MEMPTRY)(HANDLE  hDriver, uint32_t ulBoard, uint32_t ulCmd,void* pvMemoryInfo);

static uint32_t           s_ulChannelOffset     = 0;
static uint32_t           s_ulChannelSize       = 0;

static uint32_t           s_ulDPMSize           = 0;
static uint8_t*           s_pbDPM               = 0;
static MEMORY_INFORMATION s_tMemoryInfo         = {0};
static PFN_DRIVER_MEMPTRY s_pfnDriverGetMemPtr  = NULL;
static HMODULE            s_hCifxDll            = NULL;

#include "netx51_romloader_dpm.h"

static void cifXNotify(void* pvDeviceInstance, CIFX_TOOLKIT_NOTIFY_E eEvent)
{
  DEVICEINSTANCE* ptDevInst = (DEVICEINSTANCE*)pvDeviceInstance;

  switch(eEvent)
  {
  case eCIFX_TOOLKIT_EVENT_PRERESET:
    break;

  case eCIFX_TOOLKIT_EVENT_POSTRESET:
    break;

  case eCIFX_TOOLKIT_EVENT_PRE_BOOTLOADER:
    if(eCHIP_TYPE_NETX51 == ptDevInst->eChipType)
    {
      /* netX51 has invalid timing configuration, so we need to adjust it here, to be able to access
         memory starting at offset 0x100 */
      NETX51_DPM_CONFIG_AREA_T* ptDpmConfig = (NETX51_DPM_CONFIG_AREA_T*)ptDevInst->pbDPM;

      ptDpmConfig->ulDpmRdyCfg    = MSK_NX56_dpm_rdy_cfg_rdy_pol |           /* DPM is ready when external RDY-signal is high. */
                                   (1 << SRT_NX56_dpm_rdy_cfg_rdy_drv_mode); /* Push-Pull Mode */
      ptDpmConfig->ulDpmTimingCfg = 0; /* Disable setup times and filter. */

      if(g_ulTraceLevel & TRACE_LEVEL_DEBUG)
      {
        USER_Trace(ptDevInst, TRACE_LEVEL_DEBUG, "Adjusting netX51/52 DPM Timings before starting 2nd Stage Loader");
      }
    }
    break;

  case eCIFX_TOOLKIT_EVENT_POST_BOOTLOADER:
    if(eCHIP_TYPE_NETX51 == ptDevInst->eChipType)
    {
      /* Reset DPM to 8Bit mode, as SIRQ is wrongly connected on NXHX51-ETM */
      NETX51_DPM_CONFIG_AREA_T* ptDpmConfig  = (NETX51_DPM_CONFIG_AREA_T*)(ptDevInst->pbDPM + ptDevInst->ulDPMSize - sizeof(*ptDpmConfig));

      ptDpmConfig->ulDpmCfg0x0 = 0;
    }
    break;

  default:
    break;
  }
}

/*****************************************************************************/
/*! Gets a pointer to the first board from the cifX driver and stores it
*   in global variables.
*   \return true on success                                                  */
/*****************************************************************************/
bool cifXGetMemptr()
{
  bool fRet = false;

  /* We need to dynamically load the dll and function pointer, to prevent duplicate
     functions (toolkit and driver functions) at link time */
  s_hCifxDll = ::LoadLibrary(_T("cifX32dll.dll"));

  if(NULL != s_hCifxDll)
  {
    /* DLL could be dynamically loaded */
    s_pfnDriverGetMemPtr = (PFN_DRIVER_MEMPTRY)GetProcAddress(s_hCifxDll, _T("xDriverMemoryPointer"));

    if(NULL == s_pfnDriverGetMemPtr)
    {
      /* xDriverMemoryPointer function not found in cifx32dll.dll */
      ::CloseHandle(s_hCifxDll);
      s_hCifxDll = NULL;
    } else
    {
      /* xDriverMemoryPointer function successfully found */
      s_tMemoryInfo.pulMemorySize = &s_ulDPMSize;
      s_tMemoryInfo.ppvMemoryPtr  = (void**)&s_pbDPM;
      s_tMemoryInfo.ulChannel             = CIFX_NO_CHANNEL;    /* Map the whole DPM area of the board */
      s_tMemoryInfo.pulChannelStartOffset = &s_ulChannelOffset;
      s_tMemoryInfo.pulChannelSize        = &s_ulChannelSize;

      int32_t lRet = s_pfnDriverGetMemPtr(NULL, 0, CIFX_MEM_PTR_OPEN, &s_tMemoryInfo);
      
      if(CIFX_NO_ERROR == lRet)
      {
        /* Pointer successfully returned */
        fRet = true;
      }
    }
  }
  return fRet;
}

/*****************************************************************************/
/*! Return a previously acquired DPM pointer                                 */
/*****************************************************************************/
void cifXReleaseMemptr()
{
  if(NULL != s_pfnDriverGetMemPtr)
  {
    int32_t lRet = s_pfnDriverGetMemPtr(NULL, 0, CIFX_MEM_PTR_CLOSE, &s_tMemoryInfo);

    lRet = lRet;

  }

  ::FreeLibrary(s_hCifxDll);
  s_pfnDriverGetMemPtr = NULL;
}

/*****************************************************************************/
/*! Displays a hex dump on the debug console (16 bytes per line)
*   \param pbData     Pointer to dump data
*   \param ulDataLen  Length of data dump                                    */
/*****************************************************************************/
void DumpData(uint8_t* pbData, uint32_t ulDataLen)
{
  for(uint32_t ulIdx = 0; ulIdx < ulDataLen; ++ulIdx)
  {
    if(0 == (ulIdx % 16))
      printf("\r\n");

    printf("%02X ", pbData[ulIdx]);
  }
  printf("\r\n");
}

/*****************************************************************************/
/*! Dumps a rcX packet to debug console
*   \param ptPacket Pointer to packed being dumped                           */
/*****************************************************************************/
void DumpPacket(CIFX_PACKET* ptPacket)
{
  printf("Dest   : 0x%08X      ID   : 0x%08X\r\n", ptPacket->tHeader.ulDest,   ptPacket->tHeader.ulId);
  printf("Src    : 0x%08X      Sta  : 0x%08X\r\n", ptPacket->tHeader.ulSrc,    ptPacket->tHeader.ulState);
  printf("DestID : 0x%08X      Cmd  : 0x%08X\r\n", ptPacket->tHeader.ulDestId, ptPacket->tHeader.ulCmd);
  printf("SrcID  : 0x%08X      Ext  : 0x%08X\r\n", ptPacket->tHeader.ulSrcId,  ptPacket->tHeader.ulExt);
  printf("Len    : 0x%08X      Rout : 0x%08X\r\n", ptPacket->tHeader.ulLen,    ptPacket->tHeader.ulRout);

  printf("Data:");
  DumpData(ptPacket->abData, ptPacket->tHeader.ulLen);
}

/*****************************************************************************/
/*! Function to demonstrate the board/channel enumeration
*   \return CIFX_NO_ERROR on success                                         */
/*****************************************************************************/
int32_t EnumBoardDemo(void)
{
  CIFXHANDLE hDriver = NULL;
  int32_t lRet    = xDriverOpen(&hDriver);

  printf("\r\n---------- Board/Channel enumeration demo ----------\r\n");

  if(CIFX_NO_ERROR == lRet)
  {
    /* Driver/Toolkit successfully opened */
    uint32_t          ulBoard    = 0;
    BOARD_INFORMATION tBoardInfo = {0};

    /* Iterate over all boards */
    while(CIFX_NO_ERROR == xDriverEnumBoards(hDriver, ulBoard, sizeof(tBoardInfo), &tBoardInfo))
    {
      printf("Found Board %s\r\n", tBoardInfo.abBoardName);
      if(strlen( (char*)tBoardInfo.abBoardAlias) != 0)
        printf(" Alias        : %s\r\n", tBoardInfo.abBoardAlias);

      printf(" DeviceNumber : %u\r\n", tBoardInfo.tSystemInfo.ulDeviceNumber);
      printf(" SerialNumber : %u\r\n", tBoardInfo.tSystemInfo.ulSerialNumber);
      printf(" Board ID     : %u\r\n", tBoardInfo.ulBoardID);
      printf(" System Error : 0x%08X\r\n", tBoardInfo.ulSystemError);
      printf(" Channels     : %u\r\n", tBoardInfo.ulChannelCnt);
      printf(" DPM Size     : %u\r\n", tBoardInfo.ulDpmTotalSize);

      uint32_t            ulChannel    = 0;
      CHANNEL_INFORMATION tChannelInfo = {0};

      /* iterate over all channels on the current board */
      while(CIFX_NO_ERROR == xDriverEnumChannels(hDriver, ulBoard, ulChannel, sizeof(tChannelInfo), &tChannelInfo))
      {
        printf(" - Channel %u:\r\n", ulChannel);
        printf("    Firmware : %s\r\n", tChannelInfo.abFWName);
        printf("    Version  : %u.%u.%u build %u\r\n", 
               tChannelInfo.usFWMajor,
               tChannelInfo.usFWMinor,
               tChannelInfo.usFWRevision,
               tChannelInfo.usFWBuild);
        printf("    Date     : %02u/%02u/%04u\r\n", 
               tChannelInfo.bFWMonth,
               tChannelInfo.bFWDay,
               tChannelInfo.usFWYear);

        ++ulChannel;
      }

      ++ulBoard;
    }

    /* close previously opened driver */
    xDriverClose(hDriver);
  }

  printf(" State = 0x%08X\r\n", lRet);
  printf("----------------------------------------------------\r\n");

  return lRet;
}

/*****************************************************************************/
/*! Function to demonstrate system channel functionality (PacketTransfer)
*   \return CIFX_NO_ERROR on success                                         */
/*****************************************************************************/
int32_t SysdeviceDemo()
{
  CIFXHANDLE hDriver = NULL;
  int32_t    lRet    = xDriverOpen(&hDriver);

  printf("\r\n---------- System Device handling demo ----------\r\n");

  if(CIFX_NO_ERROR == lRet)
  {
    /* Driver/Toolkit successfully opened */
    CIFXHANDLE hSys = NULL;
    lRet = xSysdeviceOpen(hDriver, "cifX0", &hSys);

    if(CIFX_NO_ERROR != lRet)
    {
      printf("Error opening SystemDevice!\r\n");

    } else
    {
      SYSTEM_CHANNEL_SYSTEM_INFO_BLOCK tSysInfo = {0};
     
      /* System channel successfully opened, try to read the System Info Block */
      if( CIFX_NO_ERROR != (lRet = xSysdeviceInfo(hSys, CIFX_INFO_CMD_SYSTEM_INFO_BLOCK, sizeof(tSysInfo), &tSysInfo)))
      {
        printf("Error querying system information block\r\n");
      } else
      {
        printf("System Channel Info Block:\r\n");
        printf("DPM Size         : %u\r\n", tSysInfo.ulDpmTotalSize);
        printf("Device Number    : %u\r\n", tSysInfo.ulDeviceNumber);
        printf("Serial Number    : %u\r\n", tSysInfo.ulSerialNumber);
        printf("Manufacturer     : %u\r\n", tSysInfo.usManufacturer);
        printf("Production Date  : %u\r\n", tSysInfo.usProductionDate);
        printf("Device Class     : %u\r\n", tSysInfo.usDeviceClass);
        printf("HW Revision      : %u\r\n", tSysInfo.bHwRevision);
        printf("HW Compatibility : %u\r\n", tSysInfo.bHwCompatibility);
      }

      uint32_t ulSendPktCount = 0;
      uint32_t ulRecvPktCount = 0;

      /* Do a simple Packet exchange via system channel */
      xSysdeviceGetMBXState(hSys, &ulRecvPktCount, &ulSendPktCount);
      printf("System Mailbox State: MaxSend = %u, Pending Receive = %u\r\n",
             ulSendPktCount, ulRecvPktCount);

      CIFX_PACKET tSendPkt = {0};
      CIFX_PACKET tRecvPkt = {0};

      if(CIFX_NO_ERROR != (lRet = xSysdevicePutPacket(hSys, &tSendPkt, 10)))
      {
        printf("Error sending packet to device!\r\n");
      } else
      {
        printf("Send Packet:\r\n");
        DumpPacket(&tSendPkt);

        xSysdeviceGetMBXState(hSys, &ulRecvPktCount, &ulSendPktCount);
        printf("System Mailbox State: MaxSend = %u, Pending Receive = %u\r\n",
              ulSendPktCount, ulRecvPktCount);

        if(CIFX_NO_ERROR != (lRet = xSysdeviceGetPacket(hSys, sizeof(tRecvPkt), &tRecvPkt, 20)) )
        {
          printf("Error getting packet from device!\r\n");
        } else
        {
          printf("Received Packet:\r\n");
          DumpPacket(&tRecvPkt);

          xSysdeviceGetMBXState(hSys, &ulRecvPktCount, &ulSendPktCount);
          printf("System Mailbox State: MaxSend = %u, Pending Receive = %u\r\n",
                ulSendPktCount, ulRecvPktCount);
        }
      }
      xSysdeviceClose(hSys);
    }

    xDriverClose(hDriver);
  }

  printf(" State = 0x%08X\r\n", lRet);
  printf("----------------------------------------------------\r\n");

  return lRet;
}

/*****************************************************************************/
/*! Function to demonstrate communication channel functionality
*   Packet Transfer and I/O Data exchange
*   \return CIFX_NO_ERROR on success                                         */
/*****************************************************************************/
int32_t ChannelDemo()
{
  CIFXHANDLE  hDriver = NULL;
  int32_t     lRet    = xDriverOpen(&hDriver);

  printf("\r\n---------- Communication Channel demo ----------\r\n");

  if(CIFX_NO_ERROR == lRet)
  {
    /* Driver/Toolkit successfully opened */
    CIFXHANDLE hChannel = NULL;
    lRet = xChannelOpen(hDriver, "cifX0", 0, &hChannel);

    if(CIFX_NO_ERROR != lRet)
    {
      printf("Error opening Channel!");

    } else
    {
      CHANNEL_INFORMATION tChannelInfo = {0};

      /* Channel successfully opened, so query basic information */
      if( CIFX_NO_ERROR != (lRet = xChannelInfo(hChannel, sizeof(CHANNEL_INFORMATION), &tChannelInfo)))
      {
        printf("Error querying system information block\r\n");
      } else
      {
        printf("Communication Channel Info:\r\n");
        printf("Device Number    : %u\r\n", tChannelInfo.ulDeviceNumber);
        printf("Serial Number    : %u\r\n", tChannelInfo.ulSerialNumber);
        printf("Firmware         : %s\r\n", tChannelInfo.abFWName);
        printf("FW Version       : %u.%u.%u build %u\r\n", 
                tChannelInfo.usFWMajor,
                tChannelInfo.usFWMinor,
                tChannelInfo.usFWRevision,
                tChannelInfo.usFWBuild);
        printf("FW Date          : %02u/%02u/%04u\r\n", 
                tChannelInfo.bFWMonth,
                tChannelInfo.bFWDay,
                tChannelInfo.usFWYear);

        printf("Mailbox Size     : %u\r\n", tChannelInfo.ulMailboxSize);
      }

      CIFX_PACKET tSendPkt = {0};
      CIFX_PACKET tRecvPkt = {0};

      /* Do a basic Packet Transfer */
      if(CIFX_NO_ERROR != (lRet = xChannelPutPacket(hChannel, &tSendPkt, 10)))
      {
        printf("Error sending packet to device!\r\n");
      } else
      {
        printf("Send Packet:\r\n");
        DumpPacket(&tSendPkt);

        if(CIFX_NO_ERROR != (lRet = xChannelGetPacket(hChannel, sizeof(tRecvPkt), &tRecvPkt, 20)) )
        {
          printf("Error getting packet from device!\r\n");
        } else
        {
          printf("Received Packet:\r\n");
          DumpPacket(&tRecvPkt);
        }
      }

      /* Read and write I/O data (32Bytes). Output data will be incremented each cyle */
      uint8_t abSendData[32] = {0};
      uint8_t abRecvData[32] = {0};

      for( uint32_t ulCycle = 0; ulCycle < 10; ++ulCycle)
      {
        if(CIFX_NO_ERROR != (lRet = xChannelIORead(hChannel, 0, 0, sizeof(abRecvData), abRecvData, 10)))
        {
          printf("Error reading IO Data area!\r\n");
          break;
        } else
        {
          printf("IORead Data:");
          DumpData(abRecvData, sizeof(abRecvData));

          if(CIFX_NO_ERROR != (lRet = xChannelIOWrite(hChannel, 0, 0, sizeof(abSendData), abSendData, 10)))
          {
            printf("Error writing to IO Data area!\r\n");
            break;
          } else
          {
            printf("IOWrite Data:");
            DumpData(abSendData, sizeof(abSendData));

            memset(abSendData, ulCycle + 1, sizeof(abSendData));
          }
        }
      }

      xChannelClose(hChannel);
    }

    xDriverClose(hDriver);
  }

  printf(" State = 0x%08X\r\n", lRet);
  printf("----------------------------------------------------\r\n");

  return lRet;

}

/*****************************************************************************/
/*! Function to demonstrate control/status block functionality
*   \return CIFX_NO_ERROR on success                                         */
/*****************************************************************************/
int32_t BlockDemo( void)
{
  CIFXHANDLE hDriver = NULL;
  int32_t    lRet    = xDriverOpen(&hDriver);

  printf("\n--- Read / Write Block Information ---\r\n");  

  if(CIFX_NO_ERROR == lRet)
  {
    /* Open channel */
    CIFXHANDLE hDevice = NULL;
    lRet = xChannelOpen(hDriver, "cifX0", 0, &hDevice);

    if(lRet != CIFX_NO_ERROR)
    {
      printf("Error opening Channel!\r\n");
    } else
    {
      uint8_t abBuffer[4] = {0};

      /* Read / Write control block */
      printf("Read CONTROL Block \r\n");  
      memset( abBuffer, 0, sizeof(abBuffer));
      lRet = xChannelControlBlock( hDevice, CIFX_CMD_READ_DATA, 0, 4, &abBuffer[0]);

      DumpData(abBuffer, 4);

      printf("Write CONTROL Block \r\n");  
      lRet = xChannelControlBlock( hDevice, CIFX_CMD_WRITE_DATA, 0, 4, &abBuffer[0]);

      printf("Read COMMON Status Block \r\n");  
      memset( abBuffer, 0, sizeof(abBuffer));
      lRet = xChannelCommonStatusBlock( hDevice, CIFX_CMD_READ_DATA, 0, 4, &abBuffer[0]);

      DumpData(abBuffer, 4);

      printf("Write COMMON Status Block \r\n");  
      lRet = xChannelCommonStatusBlock( hDevice, CIFX_CMD_WRITE_DATA, 0, 4, &abBuffer[0]);

      /* this is expected to fail, as this block must not be written by Host */
      if(CIFX_NO_ERROR != lRet)
        printf("Error writing to common status block. lRet = 0x%08x\r\n", lRet);

      printf("Read EXTENDED Status Block \r\n");  
      memset( abBuffer, 0, sizeof(abBuffer));
      lRet = xChannelExtendedStatusBlock( hDevice, CIFX_CMD_READ_DATA, 0, 4, &abBuffer[0]);
      DumpData(abBuffer, 4);
      
      printf("Write EXTENDED Status Block \r\n");  
      lRet = xChannelExtendedStatusBlock( hDevice, CIFX_CMD_WRITE_DATA, 0, 4, &abBuffer[0]);

      /* this is expected to fail, as this block must not be written by Host */
      if(CIFX_NO_ERROR != lRet)
        printf("Error writing to extended status block. lRet = 0x%08x\r\n", lRet);

      xChannelClose(hDevice);
    } 
    
    xDriverClose(hDriver);
  }  

  return lRet;
}

/*****************************************************************************/
/*! Main entry function
*   \return 0                                                                */
/*****************************************************************************/
int _tmain(int /*argc*/, _TCHAR* /*argv[]*/)
{
  /* First of all initialize toolkit */
  int32_t lRet = cifXTKitInit();

  if(CIFX_NO_ERROR == lRet)
  {
    PDEVICEINSTANCE ptDevInstance = (PDEVICEINSTANCE)OS_Memalloc(sizeof(*ptDevInstance));
    OS_Memset(ptDevInstance, 0, sizeof(*ptDevInstance));

    /*Query the cifX Driver for a DPM pointer to the first board */
    if(cifXGetMemptr())
    {
      /* Insert the basic information from a prior PCI scan (or like here from the driver), into
         the DeviceInstance structure for the toolkit.
         NOTE: The physical address and irq number are for information use 
               only, so we skip them here. Interrupt is currently not supported
               and ignored, so we dont need to set it */
      ptDevInstance->fPCICard          = 0;           /* if we set true here, the card would get reset
                                                         and vanish from the PCI bus, as the PCI registers
                                                         cannot be restored from user mode */
      ptDevInstance->pvOSDependent     = NULL;
      ptDevInstance->pbDPM             = s_pbDPM;
      ptDevInstance->ulDPMSize         = s_ulDPMSize;
      ptDevInstance->eDeviceType       = eCIFX_DEVICE_AUTODETECT;
      
      ptDevInstance->pfnNotify         = cifXNotify;

      OS_Strncpy(ptDevInstance->szName,
                "cifX0",
                sizeof(ptDevInstance->szName));

      /* Add the device to the toolkits handled device list */
      lRet = cifXTKitAddDevice(ptDevInstance);

      /* If it succeeded do device tests */
      if(CIFX_NO_ERROR == lRet)
      {
        EnumBoardDemo();
        SysdeviceDemo();
        ChannelDemo();
        BlockDemo();
      }

      /* Free the pointer to the DPM again */
      cifXReleaseMemptr();
    }

    /* Uninitialize Toolkit, this will remove all handled boards from the toolkit and 
       deallocate the device instance */
    cifXTKitDeinit();
  }

	return 0;
}
